
Type TFluidDragController Extends TController
	
	Field _fluidContainer:IFluidContainer
	Field _geomList:TList
	Field _density:Float = 0
	Field _linearDragCoefficient:Float = 0
	Field _rotationalDragCoefficient:Float = 0
	Field _gravity:Vector2 = Vector2.Zero()
	Field _vertices:TVertices = New TVertices
	
	Field _totalArea:Float = 0
	Field _area:Float = 0
	Field _centroid:Vector2 = Vector2.Zero()
	Field _bouyancyForce:Vector2 = Vector2.Zero()
	Field _linearDragForce:Vector2 = Vector2.Zero()
	Field _rotationalDragTorque:Float = 0
	
	Field _geomInFluidList:TMap
	'TODO: on entry handler
	Field _fluidEntryListeners:TList = CreateList()
	
	Method New()
		_geomList = CreateList()
		_geomInfluidList = CreateMap()
	End Method
	
	Function Create:TFluidDragController(fluidContainer:IFluidContainer, density:Float, linearDragCoefficient:Float, rotationalDragCoefficient:Float, gravity:Vector2)
		Local f:TFluidDragController = New TFluidDragController
		f._fluidContainer = fluidContainer
		f._density = density
		f._linearDragCoefficient = linearDragCoefficient
		f._rotationalDragCoefficient = rotationalDragCoefficient
		f._gravity.Set(gravity)
		Return f
	End Function
	
	Method AddGeom(geom:TGeom)
		_geomList.AddLast(geom)
		_geomInfluidList.Insert(geom, Null)
	End Method
	
	
	Method AddOnEntryListener(o:Object)
		_fluidEntryListeners.AddLast(o)
	End Method
	
	Field _totalForce:Vector2 = Vector2.Zero()
	Method Update(dt:Float)
		
		For Local geom:TGeom = EachIn _geomList
			_totalArea = geom._localVertices.GetArea()
			If Not(_fluidContainer.Intersect(geom._aabb)) Then
				If _geomInFluidList.ValueForKey(geom) <> Null Then ' remove geom from in fluid list
					_geomInFluidList.Remove(geom)
					_geomInFluidList.Insert(geom, Null)
					OnExit(geom)
				End If
				Continue
			End If
			FindVerticesInFluid(geom)
			If _vertices.Count() < 3 Then
			 	Continue
			End If
			_area = _vertices.GetArea()
			If _area < 0.000001 Then Continue
			
			_centroid = _vertices.GetCentroidByArea(_area)
			
			CalculateBuoyancy()
			
			CalculateDrag(geom)
			Vector2.AddVectorsRef(_bouyancyForce, _linearDragForce, _totalForce)
			geom._body.ApplyForceAtWorldPoint(_totalForce, _centroid)
			
			geom._body.ApplyTorque(_rotationalDragTorque)
			If _geomInFluidList.ValueForKey(geom) = Null Then
				_geomInFluidList.Remove(geom)
				_geomInFluidList.Insert(geom, geom)
				'TODO: OnEntry Event handling
				OnEntry(geom)
			End If
		Next
	End Method
	
	'event handling
	Method OnEntry(geom:TGeom)
		For Local obj:Object = EachIn Self._fluidEntryListeners
			obj.SendMessage(Self, TFluidEntryEventArgs.Create(geom, True))
		Next
	End Method
	
	'event handling
	Method OnExit(geom:TGeom)
		For Local obj:Object = EachIn Self._fluidEntryListeners
			obj.SendMessage(Self, TFluidEntryEventArgs.Create(geom, False))
		Next
	End Method
	
	Field _vert:Vector2 = Vector2.Zero()
	Method FindVerticesInFluid(geom:TGeom)
		_vertices.Clear()
		For Local i:Int = 0 To geom._worldVertices.Count() - 1
			_vert = geom._worldVertices._vecArray[i]
			If _fluidContainer.Contains(_vert) Then
				_vertices.Add(_vert.Copy())
			End If			
		Next
	End Method
	
	Method CalculateAreaAndCentroid()
		_area = _vertices.GetArea()
		_centroid = _vertices.GetCentroidByArea(_area)
	End Method
	
	Method CalculateBuoyancy()
		_bouyancyForce.X = -_gravity.X * _area * _density
		_bouyancyForce.Y = -_gravity.Y * _area * _density	
	End Method
	
	
	Field _centroidVelocity:Vector2 = Vector2.Zero()
	Field _axis:Vector2 = Vector2.Zero()
	Field _min:Float
	Field _max:Float
	Field _dragArea:Float
	Field _partialMass:Float
	Field _localCentroid:Vector2 = Vector2.Zero()
	Method CalculateDrag(geom:TGeom)
		
		geom._body.GetVelocityAtWorldPointRef(_centroid, _centroidVelocity)
		
		_axis.X = -_centroidvelocity.Y
		_axis.Y = _centroidVelocity.X
		'can't normalize a zero length vector
		If _axis.X > 0 Or _axis.Y > 0 Then
			_axis.Normalize()
		End If
		
		_vertices.ProjectToAxis(_axis, _min, _max)
		_dragArea = Abs(_max - _min)
		
		_partialMass = geom._body._mass * (_area / _totalArea)
		
		_linearDragForce.X = -.5 * _density * _dragArea * _linearDragCoefficient * _partialMass * _centroidVelocity.X
		_linearDragForce.Y = -.5 * _density * _dragArea * _linearDragCoefficient * _partialMass * _centroidVelocity.Y
		
		_rotationalDragTorque = -geom._body._angularVelocity * _rotationalDragCoefficient * _partialMass		
	End Method
End Type

Type TFluidEntryEventArgs
	Field geom:TGeom
	Field isEntry:Int
	Field isExit:Int
	
	Function Create:TFluidEntryEventArgs(geom:TGeom, isEntry:Int)
		Local f:TFluidEntryEventArgs = New TFluidEntryEventArgs
		f.geom = geom
		f.isEntry = isEntry
		f.isExit = Not isEntry
		Return f
	End Function
End Type